/**

 * Creates map, draws paths, binds events.

 * @constructor

 * @param {Object} params Parameters to initialize map with.

 * @param {String} params.map Name of the map in the format <code>territory_proj_lang</code> where <code>territory</code> is a unique code or name of the territory which the map represents (ISO 3166 alpha 2 standard is used where possible), <code>proj</code> is a name of projection used to generate representation of the map on the plane (projections are named according to the conventions of proj4 utility) and <code>lang</code> is a code of the language, used for the names of regions.

 * @param {String} params.backgroundColor Background color of the map in CSS format.

 * @param {Boolean} params.zoomOnScroll When set to true map could be zoomed using mouse scroll. Default value is <code>true</code>.

 * @param {Number} params.zoomMax Indicates the maximum zoom ratio which could be reached zooming the map. Default value is <code>8</code>.

 * @param {Number} params.zoomMin Indicates the minimum zoom ratio which could be reached zooming the map. Default value is <code>1</code>.

 * @param {Number} params.zoomStep Indicates the multiplier used to zoom map with +/- buttons. Default value is <code>1.6</code>.

 * @param {Boolean} params.regionsSelectable When set to true regions of the map could be selected. Default value is <code>false</code>.

 * @param {Boolean} params.regionsSelectableOne Allow only one region to be selected at the moment. Default value is <code>false</code>.

 * @param {Boolean} params.markersSelectable When set to true markers on the map could be selected. Default value is <code>false</code>.

 * @param {Boolean} params.markersSelectableOne Allow only one marker to be selected at the moment. Default value is <code>false</code>.

 * @param {Object} params.regionStyle Set the styles for the map's regions. Each region or marker has four states: <code>initial</code> (default state), <code>hover</code> (when the mouse cursor is over the region or marker), <code>selected</code> (when region or marker is selected), <code>selectedHover</code> (when the mouse cursor is over the region or marker and it's selected simultaneously). Styles could be set for each of this states. Default value for that parameter is:

<pre>{

  initial: {

    fill: 'white',

    "fill-opacity": 1,

    stroke: 'none',

    "stroke-width": 0,

    "stroke-opacity": 1

  },

  hover: {

    "fill-opacity": 0.8

  },

  selected: {

    fill: 'yellow'

  },

  selectedHover: {

  }

}</pre>

 * @param {Object} params.markerStyle Set the styles for the map's markers. Any parameter suitable for <code>regionStyle</code> could be used as well as numeric parameter <code>r</code> to set the marker's radius. Default value for that parameter is:

<pre>{

  initial: {

    fill: 'grey',

    stroke: '#505050',

    "fill-opacity": 1,

    "stroke-width": 1,

    "stroke-opacity": 1,

    r: 5

  },

  hover: {

    stroke: 'black',

    "stroke-width": 2

  },

  selected: {

    fill: 'blue'

  },

  selectedHover: {

  }

}</pre>

 * @param {Object|Array} params.markers Set of markers to add to the map during initialization. In case of array is provided, codes of markers will be set as string representations of array indexes. Each marker is represented by <code>latLng</code> (array of two numeric values), <code>name</code> (string which will be show on marker's label) and any marker styles.

 * @param {Object} params.series Object with two keys: <code>markers</code> and <code>regions</code>. Each of which is an array of series configs to be applied to the respective map elements. See <a href="jvm.DataSeries.html">DataSeries</a> description for a list of parameters available.

 * @param {Object|String} params.focusOn This parameter sets the initial position and scale of the map viewport. It could be expressed as a string representing region which should be in focus or an object representing coordinates and scale to set. For example to focus on the center of the map at the double scale you can provide the following value:

<pre>{

  x: 0.5,

  y: 0.5,

  scale: 2

}</pre>

 * @param {Array|Object|String} params.selectedRegions Set initially selected regions.

 * @param {Array|Object|String} params.selectedMarkers Set initially selected markers.

 * @param {Function} params.onRegionLabelShow <code>(Event e, Object label, String code)</code> Will be called right before the region label is going to be shown.

 * @param {Function} params.onRegionOver <code>(Event e, String code)</code> Will be called on region mouse over event.

 * @param {Function} params.onRegionOut <code>(Event e, String code)</code> Will be called on region mouse out event.

 * @param {Function} params.onRegionClick <code>(Event e, String code)</code> Will be called on region click event.

 * @param {Function} params.onRegionSelected <code>(Event e, String code, Boolean isSelected, Array selectedRegions)</code> Will be called when region is (de)selected. <code>isSelected</code> parameter of the callback indicates whether region is selected or not. <code>selectedRegions</code> contains codes of all currently selected regions.

 * @param {Function} params.onMarkerLabelShow <code>(Event e, Object label, String code)</code> Will be called right before the marker label is going to be shown.

 * @param {Function} params.onMarkerOver <code>(Event e, String code)</code> Will be called on marker mouse over event.

 * @param {Function} params.onMarkerOut <code>(Event e, String code)</code> Will be called on marker mouse out event.

 * @param {Function} params.onMarkerClick <code>(Event e, String code)</code> Will be called on marker click event.

 * @param {Function} params.onMarkerSelected <code>(Event e, String code, Boolean isSelected, Array selectedMarkers)</code> Will be called when marker is (de)selected. <code>isSelected</code> parameter of the callback indicates whether marker is selected or not. <code>selectedMarkers</code> contains codes of all currently selected markers.

 * @param {Function} params.onViewportChange <code>(Event e, Number scale)</code> Triggered when the map's viewport is changed (map was panned or zoomed).

 */

jvm.WorldMap = function(params) {

  var map = this,

      e;



  this.params = jvm.$.extend(true, {}, jvm.WorldMap.defaultParams, params);



  if (!jvm.WorldMap.maps[this.params.map]) {

    throw new Error('Attempt to use map which was not loaded: '+this.params.map);

  }



  this.mapData = jvm.WorldMap.maps[this.params.map];

  this.markers = {};

  this.regions = {};

  this.regionsColors = {};

  this.regionsData = {};



  this.container = jvm.$('<div>').css({width: '100%', height: '100%'}).addClass('jvectormap-container');

  this.params.container.append( this.container );

  this.container.data('mapObject', this);

  this.container.css({

    position: 'relative',

    overflow: 'hidden'

  });



  this.defaultWidth = this.mapData.width;

  this.defaultHeight = this.mapData.height;



  this.setBackgroundColor(this.params.backgroundColor);



  this.onResize = function(){

    map.setSize();

  }

  jvm.$(window).resize(this.onResize);



  for (e in jvm.WorldMap.apiEvents) {

    if (this.params[e]) {

      this.container.bind(jvm.WorldMap.apiEvents[e]+'.jvectormap', this.params[e]);

    }

  }



  this.canvas = new jvm.VectorCanvas(this.container[0], this.width, this.height);



  if ( ('ontouchstart' in window) || (window.DocumentTouch && document instanceof DocumentTouch) ) {

    if (this.params.bindTouchEvents) {

      this.bindContainerTouchEvents();

    }

  } else {

    this.bindContainerEvents();

  }

  this.bindElementEvents();

  this.createLabel();

  if (this.params.zoomButtons) {

    this.bindZoomButtons();

  }

  this.createRegions();

  this.createMarkers(this.params.markers || {});



  this.setSize();



  if (this.params.focusOn) {

    if (typeof this.params.focusOn === 'object') {

      this.setFocus.call(this, this.params.focusOn.scale, this.params.focusOn.x, this.params.focusOn.y);

    } else {

      this.setFocus.call(this, this.params.focusOn);

    }

  }



  if (this.params.selectedRegions) {

    this.setSelectedRegions(this.params.selectedRegions);

  }

  if (this.params.selectedMarkers) {

    this.setSelectedMarkers(this.params.selectedMarkers);

  }



  if (this.params.series) {

    this.createSeries();

  }

};



jvm.WorldMap.prototype = {

  transX: 0,

  transY: 0,

  scale: 1,

  baseTransX: 0,

  baseTransY: 0,

  baseScale: 1,



  width: 0,

  height: 0,



  /**

   * Set background color of the map.

   * @param {String} backgroundColor Background color in CSS format.

   */

  setBackgroundColor: function(backgroundColor) {

    this.container.css('background-color', backgroundColor);

  },



  resize: function() {

    var curBaseScale = this.baseScale;

    if (this.width / this.height > this.defaultWidth / this.defaultHeight) {

      this.baseScale = this.height / this.defaultHeight;

      this.baseTransX = Math.abs(this.width - this.defaultWidth * this.baseScale) / (2 * this.baseScale);

    } else {

      this.baseScale = this.width / this.defaultWidth;

      this.baseTransY = Math.abs(this.height - this.defaultHeight * this.baseScale) / (2 * this.baseScale);

    }

    this.scale *= this.baseScale / curBaseScale;

    this.transX *= this.baseScale / curBaseScale;

    this.transY *= this.baseScale / curBaseScale;

  },



  /**

   * Synchronize the size of the map with the size of the container. Suitable in situations where the size of the container is changed programmatically or container is shown after it became visible.

   */

  setSize: function(){

    this.width = this.container.width();

    this.height = this.container.height();

    this.resize();

    this.canvas.setSize(this.width, this.height);

    this.applyTransform();

  },



  /**

   * Reset all the series and show the map with the initial zoom.

   */

  reset: function() {

    var key,

        i;



    for (key in this.series) {

      for (i = 0; i < this.series[key].length; i++) {

        this.series[key][i].clear();

      }

    }

    this.scale = this.baseScale;

    this.transX = this.baseTransX;

    this.transY = this.baseTransY;

    this.applyTransform();

  },



  applyTransform: function() {

    var maxTransX,

        maxTransY,

        minTransX,

        minTransY;



    if (this.defaultWidth * this.scale <= this.width) {

      maxTransX = (this.width - this.defaultWidth * this.scale) / (2 * this.scale);

      minTransX = (this.width - this.defaultWidth * this.scale) / (2 * this.scale);

    } else {

      maxTransX = 0;

      minTransX = (this.width - this.defaultWidth * this.scale) / this.scale;

    }



    if (this.defaultHeight * this.scale <= this.height) {

      maxTransY = (this.height - this.defaultHeight * this.scale) / (2 * this.scale);

      minTransY = (this.height - this.defaultHeight * this.scale) / (2 * this.scale);

    } else {

      maxTransY = 0;

      minTransY = (this.height - this.defaultHeight * this.scale) / this.scale;

    }



    if (this.transY > maxTransY) {

      this.transY = maxTransY;

    } else if (this.transY < minTransY) {

      this.transY = minTransY;

    }

    if (this.transX > maxTransX) {

      this.transX = maxTransX;

    } else if (this.transX < minTransX) {

      this.transX = minTransX;

    }



    this.canvas.applyTransformParams(this.scale, this.transX, this.transY);



    if (this.markers) {

      this.repositionMarkers();

    }



    this.container.trigger('viewportChange', [this.scale/this.baseScale, this.transX, this.transY]);

  },



  bindContainerEvents: function(){

    var mouseDown = false,

        oldPageX,

        oldPageY,

        map = this;



    this.container.mousemove(function(e){

      if (mouseDown) {

        map.transX -= (oldPageX - e.pageX) / map.scale;

        map.transY -= (oldPageY - e.pageY) / map.scale;



        map.applyTransform();



        oldPageX = e.pageX;

        oldPageY = e.pageY;

      }

      return false;

    }).mousedown(function(e){

      mouseDown = true;

      oldPageX = e.pageX;

      oldPageY = e.pageY;

      return false;

    });



    jvm.$('body').mouseup(function(){

      mouseDown = false;

    });



    if (this.params.zoomOnScroll) {

      this.container.mousewheel(function(event, delta, deltaX, deltaY) {

        var offset = jvm.$(map.container).offset(),

            centerX = event.pageX - offset.left,

            centerY = event.pageY - offset.top,

            zoomStep = Math.pow(1.3, deltaY);



        map.label.hide();



        map.setScale(map.scale * zoomStep, centerX, centerY);

        event.preventDefault();

      });

    }

  },



  bindContainerTouchEvents: function(){

    var touchStartScale,

        touchStartDistance,

        map = this,

        touchX,

        touchY,

        centerTouchX,

        centerTouchY,

        lastTouchesLength,

        handleTouchEvent = function(e){

          var touches = e.originalEvent.touches,

              offset,

              scale,

              transXOld,

              transYOld;



          if (e.type == 'touchstart') {

            lastTouchesLength = 0;

          }



          if (touches.length == 1) {

            if (lastTouchesLength == 1) {

              transXOld = map.transX;

              transYOld = map.transY;

              map.transX -= (touchX - touches[0].pageX) / map.scale;

              map.transY -= (touchY - touches[0].pageY) / map.scale;

              map.applyTransform();

              map.label.hide();

              if (transXOld != map.transX || transYOld != map.transY) {

                e.preventDefault();

              }

            }

            touchX = touches[0].pageX;

            touchY = touches[0].pageY;

          } else if (touches.length == 2) {

            if (lastTouchesLength == 2) {

              scale = Math.sqrt(

                Math.pow(touches[0].pageX - touches[1].pageX, 2) +

                Math.pow(touches[0].pageY - touches[1].pageY, 2)

              ) / touchStartDistance;

              map.setScale(

                touchStartScale * scale,

                centerTouchX,

                centerTouchY

              )

              map.label.hide();

              e.preventDefault();

            } else {

              offset = jvm.$(map.container).offset();

              if (touches[0].pageX > touches[1].pageX) {

                centerTouchX = touches[1].pageX + (touches[0].pageX - touches[1].pageX) / 2;

              } else {

                centerTouchX = touches[0].pageX + (touches[1].pageX - touches[0].pageX) / 2;

              }

              if (touches[0].pageY > touches[1].pageY) {

                centerTouchY = touches[1].pageY + (touches[0].pageY - touches[1].pageY) / 2;

              } else {

                centerTouchY = touches[0].pageY + (touches[1].pageY - touches[0].pageY) / 2;

              }

              centerTouchX -= offset.left;

              centerTouchY -= offset.top;

              touchStartScale = map.scale;

              touchStartDistance = Math.sqrt(

                Math.pow(touches[0].pageX - touches[1].pageX, 2) +

                Math.pow(touches[0].pageY - touches[1].pageY, 2)

              );

            }

          }



          lastTouchesLength = touches.length;

        };



    jvm.$(this.container).bind('touchstart', handleTouchEvent);

    jvm.$(this.container).bind('touchmove', handleTouchEvent);

  },



  bindElementEvents: function(){

    var map = this,

        mouseMoved;



    this.container.mousemove(function(){

      mouseMoved = true;

    });



    /* Can not use common class selectors here because of the bug in jQuery

       SVG handling, use with caution. */

    this.container.delegate("[class~='jvectormap-element']", 'mouseover mouseout', function(e){

      var path = this,

          baseVal = jvm.$(this).attr('class').baseVal ? jvm.$(this).attr('class').baseVal : jvm.$(this).attr('class'),

          type = baseVal.indexOf('jvectormap-region') === -1 ? 'marker' : 'region',

          code = type == 'region' ? jvm.$(this).attr('data-code') : jvm.$(this).attr('data-index'),

          element = type == 'region' ? map.regions[code].element : map.markers[code].element,

          labelText = type == 'region' ? map.mapData.paths[code].name : (map.markers[code].config.name || ''),

          labelShowEvent = jvm.$.Event(type+'LabelShow.jvectormap'),

          overEvent = jvm.$.Event(type+'Over.jvectormap');



      if (e.type == 'mouseover') {

        map.container.trigger(overEvent, [code]);

        if (!overEvent.isDefaultPrevented()) {

          element.setHovered(true);

        }



        map.label.text(labelText);

        map.container.trigger(labelShowEvent, [map.label, code]);

        if (!labelShowEvent.isDefaultPrevented()) {

          map.label.show();

          map.labelWidth = map.label.width();

          map.labelHeight = map.label.height();

        }

      } else {

        element.setHovered(false);

        map.label.hide();

        map.container.trigger(type+'Out.jvectormap', [code]);

      }

    });



    /* Can not use common class selectors here because of the bug in jQuery

       SVG handling, use with caution. */

    this.container.delegate("[class~='jvectormap-element']", 'mousedown', function(e){

      mouseMoved = false;

    });



    /* Can not use common class selectors here because of the bug in jQuery

       SVG handling, use with caution. */

    this.container.delegate("[class~='jvectormap-element']", 'mouseup', function(e){

      var path = this,

          baseVal = jvm.$(this).attr('class').baseVal ? jvm.$(this).attr('class').baseVal : jvm.$(this).attr('class'),

          type = baseVal.indexOf('jvectormap-region') === -1 ? 'marker' : 'region',

          code = type == 'region' ? jvm.$(this).attr('data-code') : jvm.$(this).attr('data-index'),

          clickEvent = jvm.$.Event(type+'Click.jvectormap'),

          element = type == 'region' ? map.regions[code].element : map.markers[code].element;



      if (!mouseMoved) {

        map.container.trigger(clickEvent, [code]);

        if ((type === 'region' && map.params.regionsSelectable) || (type === 'marker' && map.params.markersSelectable)) {

          if (!clickEvent.isDefaultPrevented()) {

            if (map.params[type+'sSelectableOne']) {

              map.clearSelected(type+'s');

            }

            element.setSelected(!element.isSelected);

          }

        }

      }

    });

  },



  bindZoomButtons: function() {

    var map = this;



    jvm.$('<div/>').addClass('jvectormap-zoomin').text('+').appendTo(this.container);

    jvm.$('<div/>').addClass('jvectormap-zoomout').html('&#x2212;').appendTo(this.container);



    this.container.find('.jvectormap-zoomin').click(function(){

      map.setScale(map.scale * map.params.zoomStep, map.width / 2, map.height / 2);

    });

    this.container.find('.jvectormap-zoomout').click(function(){

      map.setScale(map.scale / map.params.zoomStep, map.width / 2, map.height / 2);

    });

  },



  createLabel: function(){

    var map = this;



    this.label = jvm.$('<div/>').addClass('jvectormap-label').appendTo(jvm.$('body'));



    this.container.mousemove(function(e){

      var left = e.pageX-15-map.labelWidth,

          top = e.pageY-15-map.labelHeight;



      if (left < 5) {

        left = e.pageX + 15;

      }

      if (top < 5) {

        top = e.pageY + 15;

      }



      if (map.label.is(':visible')) {

        map.label.css({

          left: left,

          top: top

        })

      }

    });

  },



  setScale: function(scale, anchorX, anchorY, isCentered) {

    var zoomStep,

        viewportChangeEvent = jvm.$.Event('zoom.jvectormap');



    if (scale > this.params.zoomMax * this.baseScale) {

      scale = this.params.zoomMax * this.baseScale;

    } else if (scale < this.params.zoomMin * this.baseScale) {

      scale = this.params.zoomMin * this.baseScale;

    }



    if (typeof anchorX != 'undefined' && typeof anchorY != 'undefined') {

      zoomStep = scale / this.scale;

      if (isCentered) {

        this.transX = anchorX + this.defaultWidth * (this.width / (this.defaultWidth * scale)) / 2;

        this.transY = anchorY + this.defaultHeight * (this.height / (this.defaultHeight * scale)) / 2;

      } else {

        this.transX -= (zoomStep - 1) / scale * anchorX;

        this.transY -= (zoomStep - 1) / scale * anchorY;

      }

    }



    this.scale = scale;

    this.applyTransform();

    this.container.trigger(viewportChangeEvent, [scale/this.baseScale]);

  },



  /**

   * Set the map's viewport to the specific point and set zoom of the map to the specific level. Point and zoom level could be defined in two ways: using the code of some region to focus on or a central point and zoom level as numbers.

   * @param {Number|String|Array} scale|regionCode|regionCodes If the first parameter of this method is a string or array of strings and there are regions with the these codes, the viewport will be set to show all these regions. Otherwise if the first parameter is a number, the viewport will be set to show the map with provided scale.

   * @param {Number} centerX Number from 0 to 1 specifying the horizontal coordinate of the central point of the viewport.

   * @param {Number} centerY Number from 0 to 1 specifying the vertical coordinate of the central point of the viewport.

   */

  setFocus: function(scale, centerX, centerY){

    var bbox,

        itemBbox,

        newBbox,

        codes,

        i;



    if (jvm.$.isArray(scale) || this.regions[scale]) {

      if (jvm.$.isArray(scale)) {

        codes = scale;

      } else {

        codes = [scale]

      }

      for (i = 0; i < codes.length; i++) {

        if (this.regions[codes[i]]) {

          itemBbox = this.regions[codes[i]].element.getBBox();

          if (itemBbox) {

            if (typeof bbox == 'undefined') {

              bbox = itemBbox;

            } else {

              newBbox = {

                x: Math.min(bbox.x, itemBbox.x),

                y: Math.min(bbox.y, itemBbox.y),

                width: Math.max(bbox.x + bbox.width, itemBbox.x + itemBbox.width) - Math.min(bbox.x, itemBbox.x),

                height: Math.max(bbox.y + bbox.height, itemBbox.y + itemBbox.height) - Math.min(bbox.y, itemBbox.y)

              }

              bbox = newBbox;

            }

          }

        }

      }

      this.setScale(

        Math.min(this.width / bbox.width, this.height / bbox.height),

        - (bbox.x + bbox.width / 2),

        - (bbox.y + bbox.height / 2),

        true

      );

    } else {

      scale = scale * this.baseScale;

      this.setScale(scale, - centerX * this.defaultWidth, - centerY * this.defaultHeight, true);

    }

  },



  getSelected: function(type){

    var key,

        selected = [];



    for (key in this[type]) {

      if (this[type][key].element.isSelected) {

        selected.push(key);

      }

    }

    return selected;

  },



  /**

   * Return the codes of currently selected regions.

   * @returns {Array}

   */

  getSelectedRegions: function(){

    return this.getSelected('regions');

  },



  /**

   * Return the codes of currently selected markers.

   * @returns {Array}

   */

  getSelectedMarkers: function(){

    return this.getSelected('markers');

  },



  setSelected: function(type, keys){

    var i;



    if (typeof keys != 'object') {

      keys = [keys];

    }



    if (jvm.$.isArray(keys)) {

      for (i = 0; i < keys.length; i++) {

        this[type][keys[i]].element.setSelected(true);

      }

    } else {

      for (i in keys) {

        this[type][i].element.setSelected(!!keys[i]);

      }

    }

  },



  /**

   * Set or remove selected state for the regions.

   * @param {String|Array|Object} keys If <code>String</code> or <code>Array</code> the region(s) with the corresponding code(s) will be selected. If <code>Object</code> was provided its keys are  codes of regions, state of which should be changed. Selected state will be set if value is true, removed otherwise.

   */

  setSelectedRegions: function(keys){

    this.setSelected('regions', keys);

  },



  /**

   * Set or remove selected state for the markers.

   * @param {String|Array|Object} keys If <code>String</code> or <code>Array</code> the marker(s) with the corresponding code(s) will be selected. If <code>Object</code> was provided its keys are  codes of markers, state of which should be changed. Selected state will be set if value is true, removed otherwise.

   */

  setSelectedMarkers: function(keys){

    this.setSelected('markers', keys);

  },



  clearSelected: function(type){

    var select = {},

        selected = this.getSelected(type),

        i;



    for (i = 0; i < selected.length; i++) {

      select[selected[i]] = false;

    };



    this.setSelected(type, select);

  },



  /**

   * Remove the selected state from all the currently selected regions.

   */

  clearSelectedRegions: function(){

    this.clearSelected('regions');

  },



  /**

   * Remove the selected state from all the currently selected markers.

   */

  clearSelectedMarkers: function(){

    this.clearSelected('markers');

  },



  /**

   * Return the instance of WorldMap. Useful when instantiated as a jQuery plug-in.

   * @returns {WorldMap}

   */

  getMapObject: function(){

    return this;

  },



  /**

   * Return the name of the region by region code.

   * @returns {String}

   */

  getRegionName: function(code){

    return this.mapData.paths[code].name;

  },



  createRegions: function(){

    var key,

        region,

        map = this;



    for (key in this.mapData.paths) {

      region = this.canvas.addPath({

        d: this.mapData.paths[key].path,

        "data-code": key

      }, jvm.$.extend(true, {}, this.params.regionStyle));

      jvm.$(region.node).bind('selected', function(e, isSelected){

        map.container.trigger('regionSelected.jvectormap', [jvm.$(this).attr('data-code'), isSelected, map.getSelectedRegions()]);

      });

      region.addClass('jvectormap-region jvectormap-element');

      this.regions[key] = {

        element: region,

        config: this.mapData.paths[key]

      };

    }

  },



  createMarkers: function(markers) {

    var i,

        marker,

        point,

        markerConfig,

        markersArray,

        map = this;



    this.markersGroup = this.markersGroup || this.canvas.addGroup();



    if (jvm.$.isArray(markers)) {

      markersArray = markers.slice();

      markers = {};

      for (i = 0; i < markersArray.length; i++) {

        markers[i] = markersArray[i];

      }

    }



    for (i in markers) {

      markerConfig = markers[i] instanceof Array ? {latLng: markers[i]} : markers[i];

      point = this.getMarkerPosition( markerConfig );



      if (point !== false) {

        marker = this.canvas.addCircle({

          "data-index": i,

          cx: point.x,

          cy: point.y

        }, jvm.$.extend(true, {}, this.params.markerStyle, {initial: markerConfig.style || {}}), this.markersGroup);

        marker.addClass('jvectormap-marker jvectormap-element');

        jvm.$(marker.node).bind('selected', function(e, isSelected){

          map.container.trigger('markerSelected.jvectormap', [jvm.$(this).attr('data-index'), isSelected, map.getSelectedMarkers()]);

        });

        if (this.markers[i]) {

          this.removeMarkers([i]);

        }

        this.markers[i] = {element: marker, config: markerConfig};

      }

    }

  },



  repositionMarkers: function() {

    var i,

        point;



    for (i in this.markers) {

      point = this.getMarkerPosition( this.markers[i].config );

      if (point !== false) {

        this.markers[i].element.setStyle({cx: point.x, cy: point.y});

      }

    }

  },



  getMarkerPosition: function(markerConfig) {

    if (jvm.WorldMap.maps[this.params.map].projection) {

      return this.latLngToPoint.apply(this, markerConfig.latLng || [0, 0]);

    } else {

      return {

        x: markerConfig.coords[0]*this.scale + this.transX*this.scale,

        y: markerConfig.coords[1]*this.scale + this.transY*this.scale

      };

    }

  },



  /**

   * Add one marker to the map.

   * @param {String} key Marker unique code.

   * @param {Object} marker Marker configuration parameters.

   * @param {Array} seriesData Values to add to the data series.

   */

  addMarker: function(key, marker, seriesData){

    var markers = {},

        data = [],

        values,

        i,

        seriesData = seriesData || [];



    markers[key] = marker;



    for (i = 0; i < seriesData.length; i++) {

      values = {};

      values[key] = seriesData[i];

      data.push(values);

    }

    this.addMarkers(markers, data);

  },



  /**

   * Add set of marker to the map.

   * @param {Object|Array} markers Markers to add to the map. In case of array is provided, codes of markers will be set as string representations of array indexes.

   * @param {Array} seriesData Values to add to the data series.

   */

  addMarkers: function(markers, seriesData){

    var i;



    seriesData = seriesData || [];



    this.createMarkers(markers);

    for (i = 0; i < seriesData.length; i++) {

      this.series.markers[i].setValues(seriesData[i] || {});

    };

  },



  /**

   * Remove some markers from the map.

   * @param {Array} markers Array of marker codes to be removed.

   */

  removeMarkers: function(markers){

    var i;



    for (i = 0; i < markers.length; i++) {

      this.markers[ markers[i] ].element.remove();

      delete this.markers[ markers[i] ];

    };

  },



  /**

   * Remove all markers from the map.

   */

  removeAllMarkers: function(){

    var i,

        markers = [];



    for (i in this.markers) {

      markers.push(i);

    }

    this.removeMarkers(markers)

  },



  /**

   * Converts coordinates expressed as latitude and longitude to the coordinates in pixels on the map.

   * @param {Number} lat Latitide of point in degrees.

   * @param {Number} lng Longitude of point in degrees.

   */

  latLngToPoint: function(lat, lng) {

    var point,

        proj = jvm.WorldMap.maps[this.params.map].projection,

        centralMeridian = proj.centralMeridian,

        width = this.width - this.baseTransX * 2 * this.baseScale,

        height = this.height - this.baseTransY * 2 * this.baseScale,

        inset,

        bbox,

        scaleFactor = this.scale / this.baseScale;



    if (lng < (-180 + centralMeridian)) {

      lng += 360;

    }



    point = jvm.Proj[proj.type](lat, lng, centralMeridian);



    inset = this.getInsetForPoint(point.x, point.y);

    if (inset) {

      bbox = inset.bbox;



      point.x = (point.x - bbox[0].x) / (bbox[1].x - bbox[0].x) * inset.width * this.scale;

      point.y = (point.y - bbox[0].y) / (bbox[1].y - bbox[0].y) * inset.height * this.scale;



      return {

        x: point.x + this.transX*this.scale + inset.left*this.scale,

        y: point.y + this.transY*this.scale + inset.top*this.scale

      };

     } else {

       return false;

     }

  },



  /**

   * Converts cartesian coordinates into coordinates expressed as latitude and longitude.

   * @param {Number} x X-axis of point on map in pixels.

   * @param {Number} y Y-axis of point on map in pixels.

   */

  pointToLatLng: function(x, y) {

    var proj = jvm.WorldMap.maps[this.params.map].projection,

        centralMeridian = proj.centralMeridian,

        insets = jvm.WorldMap.maps[this.params.map].insets,

        i,

        inset,

        bbox,

        nx,

        ny;



    for (i = 0; i < insets.length; i++) {

      inset = insets[i];

      bbox = inset.bbox;



      nx = x - (this.transX*this.scale + inset.left*this.scale);

      ny = y - (this.transY*this.scale + inset.top*this.scale);



      nx = (nx / (inset.width * this.scale)) * (bbox[1].x - bbox[0].x) + bbox[0].x;

      ny = (ny / (inset.height * this.scale)) * (bbox[1].y - bbox[0].y) + bbox[0].y;



      if (nx > bbox[0].x && nx < bbox[1].x && ny > bbox[0].y && ny < bbox[1].y) {

        return jvm.Proj[proj.type + '_inv'](nx, -ny, centralMeridian);

      }

    }



    return false;

  },



  getInsetForPoint: function(x, y){

    var insets = jvm.WorldMap.maps[this.params.map].insets,

        i,

        bbox;



    for (i = 0; i < insets.length; i++) {

      bbox = insets[i].bbox;

      if (x > bbox[0].x && x < bbox[1].x && y > bbox[0].y && y < bbox[1].y) {

        return insets[i];

      }

    }

  },



  createSeries: function(){

    var i,

        key;



    this.series = {

      markers: [],

      regions: []

    };



    for (key in this.params.series) {

      for (i = 0; i < this.params.series[key].length; i++) {

        this.series[key][i] = new jvm.DataSeries(

          this.params.series[key][i],

          this[key]

        );

      }

    }

  },



  /**

   * Gracefully remove the map and and all its accessories, unbind event handlers.

   */

  remove: function(){

    this.label.remove();

    this.container.remove();

    jvm.$(window).unbind('resize', this.onResize);

  }

};



jvm.WorldMap.maps = {};

jvm.WorldMap.defaultParams = {

  map: 'world_mill_en',

  backgroundColor: '#505050',

  zoomButtons: true,

  zoomOnScroll: true,

  zoomMax: 8,

  zoomMin: 1,

  zoomStep: 1.6,

  regionsSelectable: false,

  markersSelectable: false,

  bindTouchEvents: true,

  regionStyle: {

    initial: {

      fill: 'white',

      "fill-opacity": 1,

      stroke: 'none',

      "stroke-width": 0,

      "stroke-opacity": 1

    },

    hover: {

      "fill-opacity": 0.8

    },

    selected: {

      fill: 'yellow'

    },

    selectedHover: {

    }

  },

  markerStyle: {

    initial: {

      fill: 'grey',

      stroke: '#505050',

      "fill-opacity": 1,

      "stroke-width": 1,

      "stroke-opacity": 1,

      r: 5

    },

    hover: {

      stroke: 'black',

      "stroke-width": 2

    },

    selected: {

      fill: 'blue'

    },

    selectedHover: {

    }

  }

};

jvm.WorldMap.apiEvents = {

  onRegionLabelShow: 'regionLabelShow',

  onRegionOver: 'regionOver',

  onRegionOut: 'regionOut',

  onRegionClick: 'regionClick',

  onRegionSelected: 'regionSelected',

  onMarkerLabelShow: 'markerLabelShow',

  onMarkerOver: 'markerOver',

  onMarkerOut: 'markerOut',

  onMarkerClick: 'markerClick',

  onMarkerSelected: 'markerSelected',

  onViewportChange: 'viewportChange'

};

